<?php
namespace VM\FinancialStatementsBundle\Services;

use VM\FinancialStatementsBundle\Services\DataStructure\SprawozdaniaDS;

class Sprawozdania extends CommonMethods
{
    protected $FinansowanieProdukcji;
    protected $PodsumowanieFinansowaniaInwestycji;

    protected $Aktywa;
    protected $Pasywa;
    protected $RachunekZyskowIStrat;
    protected $RachunekPrzeplywowPienieznych;
    protected $WskaznikiFinansowe;

    /**
     * @var float
     */
    protected $kapitalPodstawowy = 0.00;

    /**
     * @var int
     */
    protected $rotacjaZapasowWDniach = 14;

    /**
     * @var int
     */
    protected $rotacjaNaleznosciWDniach = 60;

    /**
     * @var int
     */
    protected $rotacjaZobowiazanWDniach = 30;

    /**
     * @var array|float
     */
    protected $przychodyFinansowe = array();

    protected $lokatyWziete = array();
    protected $lokatySplacone = array();

    /**
     * @var array|float
     */
    protected $kredytObrotowy = array();

    /**
     * @var array|float
     */
    protected $odsetkiKredytuObrotowego = array();

    /**
     * @var int
     */
    protected $doKtoregoRoku = 1;

    /**
     * @var int
     */
    protected $rokPoczatkowy = 1;

    /**
     * @var int
     */
    protected $rozmiarTablic = 1;

    /**
     * @param SprawozdaniaDS $SprawozdaniaDS
     * @return Sprawozdania
     * @throws \Exception
     */
    public static function create(SprawozdaniaDS $SprawozdaniaDS)
    {
        if($SprawozdaniaDS->validate())
        {
            return new Sprawozdania($SprawozdaniaDS);
        }
        else
        {
            throw new \Exception('Dane wejściowe nie przeszły walidacji.');
        }
    }
    /**
     * constructor
     */
    protected function __construct(SprawozdaniaDS $SprawozdaniaDS)
    {
        $this->FinansowanieProdukcji = $SprawozdaniaDS->FinansowanieProdukcji;
        $this->PodsumowanieFinansowaniaInwestycji = $SprawozdaniaDS->PodsumowanieFinansowaniaInwestycji;
//        $this->LokatyPodsumowanie = $SprawozdaniaDS->LokatyPodsumowanie;
        $this->rokPoczatkowy = $SprawozdaniaDS->rokPoczatkowy;
        $this->doKtoregoRoku = $SprawozdaniaDS->ileIteracji + $this->rokPoczatkowy;
        $this->rozmiarTablic = $SprawozdaniaDS->ileIteracji;
        $this->kapitalPodstawowy = $SprawozdaniaDS->kapitalPodstawowy;
        $this->przychodyFinansowe = $SprawozdaniaDS->przychodyFinansowe;
        $this->lokatyWziete = $SprawozdaniaDS->lokatyWziete;
        $this->lokatySplacone = $SprawozdaniaDS->lokatySplacone;
        $this->kredytObrotowy = $SprawozdaniaDS->kredytObrotowy;
        $this->kredytObrotowySplata = $SprawozdaniaDS->kredytObrotowySplata;
        $this->odsetkiKredytuObrotowego = $SprawozdaniaDS->odsetkiKredytuObrotowego;

        $this->buildAktywa();
        $this->buildPasywa();
        $this->buildRachunekZyskowIStrat();
        $this->buildRachunekPrzeplywowPienieznych();
        $this->buildWskaznikiFinansowe();

        $this->evaluateAll();

    }

    protected function sumaLokatWzietych($doRoku)
    {
        $sum = 0;
        for($i = $this->rokPoczatkowy; $i < $doRoku; $i++)
        {
            $sum += $this->lokatyWziete[$i];
        }
//        echo $sum."<br>";
        return $sum;
    }

    protected function sumaLokatSplaconych($doRoku)
    {
        $sum = 0;
        for($i = $this->rokPoczatkowy; $i < $doRoku; $i++)
        {
            $sum += $this->lokatySplacone[$i];
        }
//        echo $sum."<br>";
        return $sum;
    }

    protected function buildEmptyArrays()
    {
        return array_fill($this->rokPoczatkowy, $this->rozmiarTablic, 0);
    }

    protected function buildAktywa()
    {
        $aktywa = new \stdClass();

        $aktywa->trwale = new \stdClass();
        $aktywa->trwale->razem = $this->buildEmptyArrays();
        $aktywa->trwale->rzeczowe = $this->buildEmptyArrays();
		$aktywa->trwale->inwestycjeDlugoterminowe = $this->buildEmptyArrays();

        $aktywa->obrotowe = new \stdClass();
        $aktywa->obrotowe->razem = $this->buildEmptyArrays();
        $aktywa->obrotowe->zapasy = $this->buildEmptyArrays();
        $aktywa->obrotowe->naleznosciKrotkoterminowe = new \stdClass();
        $aktywa->obrotowe->naleznosciKrotkoterminowe->razem = $this->buildEmptyArrays();
        $aktywa->obrotowe->naleznosciKrotkoterminowe->ztytDiU = $this->buildEmptyArrays();
        $aktywa->obrotowe->inwestycjeKrotkoterminowe = $this->buildEmptyArrays();

        $aktywa->razem = $this->buildEmptyArrays();

        $this->Aktywa = $aktywa;
    }

    protected function evaluateAktywa()
    {
        $aktywa = $this->Aktywa;
        $pasywa = $this->Pasywa;
        $rachZyskow = $this->RachunekZyskowIStrat;
        $rachPrzeplywow = $this->RachunekPrzeplywowPienieznych;
        $inwestycje = $this->PodsumowanieFinansowaniaInwestycji->getPodsumowanieWszystkichInwestycji();

        $this->evaluateAktywaTrwale($aktywa, $inwestycje);

        $this->evaluateAktywaObrotoweNaleznosci($aktywa, $rachZyskow, $rachPrzeplywow);

        $this->evaluateAktywaObrotoweZapasy($aktywa, $rachZyskow, $rachPrzeplywow);

        $this->evaluateAktywaRazem($aktywa, $pasywa);
    }

    protected function evaluateAktywaTrwale($aktywa, $inwestycje)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $aktywa->trwale->rzeczowe[$i] +=
                isset($inwestycje->wartoscKoncowaNaRok[$i])? $inwestycje->wartoscKoncowaNaRok[$i]: 0;
            $aktywa->trwale->inwestycjeDlugoterminowe[$i] += $this->sumaLokatWzietych($i+1) - $this->sumaLokatSplaconych($i+1);
			$aktywa->trwale->razem[$i] += $aktywa->trwale->rzeczowe[$i] + $aktywa->trwale->inwestycjeDlugoterminowe[$i];
        }
    }

    protected function evaluateAktywaObrotoweNaleznosci($aktywa, $rachZyskow, $rachPrzeplywow)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $aktywa->obrotowe->naleznosciKrotkoterminowe->ztytDiU[$i] = 
				$rachZyskow->przychodyNettoZeSprzedazy->produktow[$i] * 2 / 12;
				
//				$i == $this->rokPoczatkowy? 0: $aktywa->obrotowe->naleznosciKrotkoterminowe->ztytDiU[$i-1];
//                $this->lokatyWziete[$i]
//                + $this->sumaLokatWzietych($i+1) - $this->sumaLokatSplaconych($i+1);

			$aktywa->obrotowe->naleznosciKrotkoterminowe->razem[$i] +=
                $aktywa->obrotowe->naleznosciKrotkoterminowe->ztytDiU[$i];
//                +
//                $this->rotacjaNaleznosciWDniach * 1.23 *
//                $rachZyskow->przychodyNettoZeSprzedazy->razem[$i] / 365;
        }
    }

    protected function evaluateAktywaObrotoweZapasy($aktywa, $rachZyskow, $rachPrzeplywow)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $liczbaZapasowNowa = $i == $this->rokPoczatkowy?
                $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i)->zapasy:
                $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i)->zapasy -
                $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i-1)->zapasy;

			$sumaKosztowRodzajowych = $rachZyskow->kosztyDzialalnosciOperacyjnej->razem[$i];
            $produkcjaObecna = $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i)->produkcjaLacznie;
			$wartoscJednZapasowNowych = $produkcjaObecna == 0 ? 0 : $sumaKosztowRodzajowych/$produkcjaObecna;
			$wartoscJednZapasowStarych = $i == $this->rokPoczatkowy ? 0 : $aktywa->obrotowe->zapasy[$i-1]/$this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i-1)->zapasy;
				
            $wartoscZapasowNowych = $liczbaZapasowNowa == 0 ? 0:
			($liczbaZapasowNowa < 0 ? $liczbaZapasowNowa * $wartoscJednZapasowStarych : $liczbaZapasowNowa * $wartoscJednZapasowNowych);
			
//            $wartoscZapasowNowych = $liczbaZapasowNowa < 1? 0:		TUTAJ POPRZEDNI KOD
//                $liczbaZapasowNowa * ($produkcjaObecna < 1? 0: $sumaKosztowRodzajowych/$produkcjaObecna);
//            print_r($produkcjaObecna);
//            echo "<br>";
//            print_r($liczbaZapasowNowa);
//            echo "<br>";
//            print_r($sumaKosztowRodzajowych);
//            echo "<br>";
//            print_r($wartoscZapasowNowych);
//            echo "<br>";
//            print_r($aktywa->obrotowe->zapasy[$i]);

            $aktywa->obrotowe->zapasy[$i] =
                $i == $this->rokPoczatkowy? $wartoscZapasowNowych:
                    $aktywa->obrotowe->zapasy[$i-1] + $wartoscZapasowNowych;
//            $aktywa->obrotowe->zapasy[$i] =
//                (($this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i)->zapasy) *
//                    $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i)->kosztProdukcjiSztuki);
        }
    }

    protected function evaluateAktywaObrotoweInwestycjeKrotkoterminowe($aktywa, $rachPrzeplywow)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $aktywa->obrotowe->inwestycjeKrotkoterminowe[$i] +=
//                $this->lokatySplacone[$i] +
                $rachPrzeplywow->srodkiNaKoniecOkresu[$i];
        }
    }

    protected function evaluateAktywaObrotoweRazem($aktywa)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $aktywa->obrotowe->razem[$i] +=
                $aktywa->obrotowe->zapasy[$i] +
                $aktywa->obrotowe->naleznosciKrotkoterminowe->razem[$i] +
                $aktywa->obrotowe->inwestycjeKrotkoterminowe[$i];
        }
    }

    protected function evaluateAktywaRazem($aktywa, $pasywa)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $aktywa->razem[$i] += $aktywa->trwale->razem[$i] + $aktywa->obrotowe->razem[$i];
        }
    }

    protected function buildPasywa()
    {
        $pasywa = new \stdClass();

        $pasywa->kapitalWlasny = new \stdClass();
        $pasywa->kapitalWlasny->razem = $this->buildEmptyArrays();
        $pasywa->kapitalWlasny->podstawowy = $this->buildEmptyArrays();
        $pasywa->kapitalWlasny->zyskZLatUbieglych = $this->buildEmptyArrays();
        $pasywa->kapitalWlasny->zyskNetto = $this->buildEmptyArrays();

        $pasywa->zobowiazania = new \stdClass();
        $pasywa->zobowiazania->razem = $this->buildEmptyArrays();
        $pasywa->zobowiazania->dlugoterminowe = new \stdClass();
        $pasywa->zobowiazania->dlugoterminowe->razem = $this->buildEmptyArrays();
        $pasywa->zobowiazania->dlugoterminowe->kredytyIPozyczki = $this->buildEmptyArrays();
        $pasywa->zobowiazania->krotkoterminowe = new \stdClass();
        $pasywa->zobowiazania->krotkoterminowe->razem = $this->buildEmptyArrays();
        $pasywa->zobowiazania->krotkoterminowe->dostawyIUslugi = $this->buildEmptyArrays();
//        $pasywa->zobowiazania->miedzyokresowe = new \stdClass();
//        $pasywa->zobowiazania->miedzyokresowe->razem = $this->buildEmptyArrays();

        $pasywa->razem = $this->buildEmptyArrays();

        $this->Pasywa = $pasywa;
    }

    protected function evaluatePasywa()
    {
        $pasywa = $this->Pasywa;
        $rachZyskow = $this->RachunekZyskowIStrat;
        $inwestycje = $this->PodsumowanieFinansowaniaInwestycji->getPodsumowanieWszystkichInwestycji();
        $aktywa = $this->Aktywa;

        $this->evaluatePasywaKapitalWlasny($pasywa, $rachZyskow);

        $this->evaluatePasywaZobowiazania($pasywa, $rachZyskow, $inwestycje, $aktywa);

        $this->evaluatePasywaRazem($pasywa);
    }

    protected function evaluatePasywaKapitalWlasny($pasywa, $rachZyskow)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
			$pasywa->kapitalWlasny->podstawowy[$i] = $this->kapitalPodstawowy;
            $pasywa->kapitalWlasny->zyskNetto[$i] =
//                $i == $this->rokPoczatkowy? 0:
                    $rachZyskow->zyskNetto[$i];
            $pasywa->kapitalWlasny->zyskZLatUbieglych[$i] =
                $i < $this->rokPoczatkowy+1? 0:
                    $pasywa->kapitalWlasny->zyskZLatUbieglych[$i-1] + $pasywa->kapitalWlasny->zyskNetto[$i-1];
            $pasywa->kapitalWlasny->razem[$i] =
                $pasywa->kapitalWlasny->podstawowy[$i] +
                $pasywa->kapitalWlasny->zyskNetto[$i] +
                $pasywa->kapitalWlasny->zyskZLatUbieglych[$i];
        }
    }

    protected function evaluatePasywaZobowiazania($pasywa, $rachZyskow, $inwestycje, $aktywa)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $zmianaKapitaluObcegoNaRok =
                isset($inwestycje->zmianaKapitaluObcegoNaRok[$i])? $inwestycje->zmianaKapitaluObcegoNaRok[$i]: 0;
            $pasywa->zobowiazania->dlugoterminowe->kredytyIPozyczki[$i] =
                $i == $this->rokPoczatkowy? $zmianaKapitaluObcegoNaRok + $this->kredytObrotowy[$i]['value']:
                $pasywa->zobowiazania->dlugoterminowe->kredytyIPozyczki[$i-1] + $zmianaKapitaluObcegoNaRok + $this->kredytObrotowy[$i]['value'] - $this->kredytObrotowySplata[$i]['fullValue'];
            $pasywa->zobowiazania->dlugoterminowe->razem[$i] = $pasywa->zobowiazania->dlugoterminowe->kredytyIPozyczki[$i];

			$produkcja = $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i);
            $pasywa->zobowiazania->krotkoterminowe->dostawyIUslugi[$i] = 
				$produkcja->kosztyZmienneKwota['materialowe'] * 1 / 12;

			
// póki co zeruje, bo to co było było bezsensu, trzeba poprawić należności i zobowiązania
//                $rachZyskow->kosztyDzialalnosciOperacyjnej->amortyzacja[$i]
//                + $aktywa->obrotowe->zapasy[$i]

//                ($rachZyskow->kosztyDzialalnosciOperacyjnej->razem[$i] -
//                $rachZyskow->kosztyDzialalnosciOperacyjnej->amortyzacja[$i]) * 1.23 *
//                $this->rotacjaZobowiazanWDniach/365;
            $pasywa->zobowiazania->krotkoterminowe->kredytyIPozyczki[$i] = 0;
//                $this->kredytObrotowy[$i];
            $pasywa->zobowiazania->krotkoterminowe->pozostale[$i] = 0;
            $pasywa->zobowiazania->krotkoterminowe->razem[$i] =
                $pasywa->zobowiazania->krotkoterminowe->dostawyIUslugi[$i]
                + $pasywa->zobowiazania->krotkoterminowe->kredytyIPozyczki[$i]
                + $pasywa->zobowiazania->krotkoterminowe->pozostale[$i];

            $pasywa->zobowiazania->razem[$i] = $pasywa->zobowiazania->dlugoterminowe->razem[$i] + $pasywa->zobowiazania->krotkoterminowe->razem[$i];
        }
    }

    protected function evaluatePasywaRazem($pasywa)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $pasywa->razem[$i] += $pasywa->kapitalWlasny->razem[$i] + $pasywa->zobowiazania->razem[$i];
        }
    }

    protected function buildRachunekZyskowIStrat()
    {
        $rachunek = new \stdClass();

        $rachunek->przychodyNettoZeSprzedazy = new \stdClass();
        $rachunek->przychodyNettoZeSprzedazy->razem = $this->buildEmptyArrays();
        $rachunek->przychodyNettoZeSprzedazy->produktow = $this->buildEmptyArrays();

        $rachunek->kosztyDzialalnosciOperacyjnej = new \stdClass();
        $rachunek->kosztyDzialalnosciOperacyjnej->razem = $this->buildEmptyArrays();
        $rachunek->kosztyDzialalnosciOperacyjnej->amortyzacja = $this->buildEmptyArrays();
        $rachunek->kosztyDzialalnosciOperacyjnej->zuzycieMaterialowIEnergii = $this->buildEmptyArrays();
        $rachunek->kosztyDzialalnosciOperacyjnej->uslugiObce = $this->buildEmptyArrays();
        $rachunek->kosztyDzialalnosciOperacyjnej->podatkiIOplaty = $this->buildEmptyArrays();
        $rachunek->kosztyDzialalnosciOperacyjnej->wynagrodzenia = $this->buildEmptyArrays();
        $rachunek->kosztyDzialalnosciOperacyjnej->ubezpieczeniaSpoleczneISwiadczenia = $this->buildEmptyArrays();

        $rachunek->zyskZeSprzedazy = $this->buildEmptyArrays();
        $rachunek->zyskZDzialalnosciOperacyjnej = $this->buildEmptyArrays();
        $rachunek->przychodyFinansowe = $this->buildEmptyArrays();
        $rachunek->kosztyFinansowe = $this->buildEmptyArrays();
        $rachunek->zyskBrutto = $this->buildEmptyArrays();
        $rachunek->podatekDochodowy = $this->buildEmptyArrays();
        $rachunek->zyskNetto = $this->buildEmptyArrays();

        $this->RachunekZyskowIStrat = $rachunek;
    }

    protected function evaluateRachunekZyskowIStrat()
    {
        $rachZyskow = $this->RachunekZyskowIStrat;
        $inwestycje = $this->PodsumowanieFinansowaniaInwestycji->getPodsumowanieWszystkichInwestycji();

        $this->evaluateRachunekZyskowPrzychodyNettoZeSprzedazy($this->Aktywa, $rachZyskow);

        $this->evaluateRachunekZyskowKosztyDzialalnosciOperacyjnej($rachZyskow, $inwestycje);

        $this->evaluateRachunekZyskowResztaOdCDoLZExcela($rachZyskow, $inwestycje);
    }

    protected function evaluateRachunekZyskowPrzychodyNettoZeSprzedazy($aktywa, $rachZyskow)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $rachZyskow->przychodyNettoZeSprzedazy->produktow[$i] =
//                $i == $this->rokPoczatkowy? 0:
                $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i)->nowyUdzialWRynku;

            $sumaKosztowRodzajowych = $rachZyskow->kosztyDzialalnosciOperacyjnej->razem[$i];
            $sprzedaz = $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i)->sprzedazLacznie;
            $produkcja = $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i)->produkcjaLacznie;
            $rachZyskow->przychodyNettoZeSprzedazy->zmianaStanuZapasow[$i] =
				$i == $this->rokPoczatkowy ? $aktywa->obrotowe->zapasy[$i]:
                ($aktywa->obrotowe->zapasy[$i] - $aktywa->obrotowe->zapasy[$i-1]);			
//				$produkcja < 1? 0:
//              $sumaKosztowRodzajowych - $sprzedaz * ($sumaKosztowRodzajowych / $produkcja);

            $rachZyskow->przychodyNettoZeSprzedazy->razem[$i] =
                $rachZyskow->przychodyNettoZeSprzedazy->produktow[$i]
                +
                $rachZyskow->przychodyNettoZeSprzedazy->zmianaStanuZapasow[$i]
            ;
        }
    }

    protected function evaluateRachunekZyskowKosztyDzialalnosciOperacyjnej($rachZyskow, $inwestycje)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $produkcja = $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($i);
            $rachZyskow->kosztyDzialalnosciOperacyjnej->amortyzacja[$i] =
//                $i == $this->rokPoczatkowy? 0:
                    (isset($inwestycje->amortyzacjaNaRok[$i])? $inwestycje->amortyzacjaNaRok[$i]: 0);
            $rachZyskow->kosztyDzialalnosciOperacyjnej->zuzycieMaterialowIEnergii[$i] =
//                $i == $this->rokPoczatkowy? 0:
                (isset($produkcja->kosztyStale['energia'])? $produkcja->kosztyStale['energia']: 0) +
                (isset($produkcja->kosztyStale['materialowe'])? $produkcja->kosztyStale['materialowe']: 0) +
                (isset($produkcja->kosztyZmienneKwota['energia'])? $produkcja->kosztyZmienneKwota['energia']: 0) +
                (isset($produkcja->kosztyZmienneKwota['materialowe'])? $produkcja->kosztyZmienneKwota['materialowe']: 0) +
                (isset($produkcja->kosztyZmienneKwota['woda'])? $produkcja->kosztyZmienneKwota['woda']: 0);
            $rachZyskow->kosztyDzialalnosciOperacyjnej->uslugiObce[$i] =
//                $i == $this->rokPoczatkowy? 0:
                (isset($produkcja->kosztyStale['telekomunikacja'])? $produkcja->kosztyStale['telekomunikacja']: 0) +
                (isset($produkcja->kosztyStale['czynsz'])? $produkcja->kosztyStale['czynsz']: 0) +
                (isset($produkcja->kosztyZmienneKwota['kosztyLogistyki'])? $produkcja->kosztyZmienneKwota['kosztyLogistyki']: 0) +
                (isset($produkcja->kosztyStale['uslugiObce'])? $produkcja->kosztyStale['uslugiObce']: 0);
            $rachZyskow->kosztyDzialalnosciOperacyjnej->podatkiIOplaty[$i] =
//                $i == $this->rokPoczatkowy? 0:
                (isset($produkcja->kosztyStale['ubezpieczenia'])? $produkcja->kosztyStale['ubezpieczenia']: 0) +
                (isset($produkcja->kosztyStale['podatkiLokalne'])? $produkcja->kosztyStale['podatkiLokalne']: 0) +
                (isset($produkcja->kosztyStale['oplatyBankowe'])? $produkcja->kosztyStale['oplatyBankowe']: 0);

            $rachZyskow->kosztyDzialalnosciOperacyjnej->wynagrodzenia[$i] =
                ((isset($produkcja->kosztyStale['osobowe'])? $produkcja->kosztyStale['osobowe']: 0)/1.7);
            $rachZyskow->kosztyDzialalnosciOperacyjnej->ubezpieczeniaSpoleczneISwiadczenia[$i] =
                ((isset($produkcja->kosztyStale['osobowe'])? $produkcja->kosztyStale['osobowe']: 0)
                    - $produkcja->kosztyStale['osobowe']/1.7);


            $rachZyskow->kosztyDzialalnosciOperacyjnej->razem[$i] =
//                $i == $this->rokPoczatkowy? 0:
                    $rachZyskow->kosztyDzialalnosciOperacyjnej->amortyzacja[$i] +
                    $rachZyskow->kosztyDzialalnosciOperacyjnej->zuzycieMaterialowIEnergii[$i] +
                    $rachZyskow->kosztyDzialalnosciOperacyjnej->uslugiObce[$i] +
                    $rachZyskow->kosztyDzialalnosciOperacyjnej->podatkiIOplaty[$i] +
                    $rachZyskow->kosztyDzialalnosciOperacyjnej->wynagrodzenia[$i] +
                    $rachZyskow->kosztyDzialalnosciOperacyjnej->ubezpieczeniaSpoleczneISwiadczenia[$i];
        }
    }

    protected function evaluateRachunekZyskowResztaOdCDoLZExcela($rachZyskow, $inwestycje)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $rachZyskow->zyskZeSprzedazy[$i] =
                $rachZyskow->przychodyNettoZeSprzedazy->razem[$i] -           ////////////////////////////////////////
                $rachZyskow->kosztyDzialalnosciOperacyjnej->razem[$i];
            $rachZyskow->zyskZDzialalnosciOperacyjnej[$i] = $rachZyskow->zyskZeSprzedazy[$i];
            $rachZyskow->przychodyFinansowe[$i] = $this->przychodyFinansowe[$i];
            $rachZyskow->kosztyFinansowe[$i] =
                -(isset($inwestycje->kosztyFinansoweNaRok[$i])? $inwestycje->kosztyFinansoweNaRok[$i]: 0) +
                $this->odsetkiKredytuObrotowego[$i];
            $rachZyskow->zyskBrutto[$i] =
                $rachZyskow->zyskZDzialalnosciOperacyjnej[$i] +
                $rachZyskow->przychodyFinansowe[$i] -
                $rachZyskow->kosztyFinansowe[$i];
            $rachZyskow->podatekDochodowy[$i] = $rachZyskow->zyskBrutto[$i] <= 0? 0: round($rachZyskow->zyskBrutto[$i]*0.19);
            $rachZyskow->zyskNetto[$i] = $rachZyskow->zyskBrutto[$i] - $rachZyskow->podatekDochodowy[$i];
        }
    }

    protected function buildRachunekPrzeplywowPienieznych()
    {
        $rachunek = new \stdClass();

        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej = new \stdClass();
        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej->razem = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej->zyskNetto = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej->korekty = new \stdClass();
        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->razem = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->amortyzacja = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->zmianaStanuZapasow = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->zmianaStanuNaleznosci = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->zmianaStanuZobowiazanKrotkoterminowych = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->inneKorekty = $this->buildEmptyArrays();

        $rachunek->przeplywySrodkowZDzialalnosciInwestycyjnej = new \stdClass();
        $rachunek->przeplywySrodkowZDzialalnosciInwestycyjnej->razem = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciInwestycyjnej->nakladyInwestycyjne = $this->buildEmptyArrays();

        $rachunek->przeplywySrodkowZDzialalnosciFinansowej = new \stdClass();
        $rachunek->przeplywySrodkowZDzialalnosciFinansowej->razem = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciFinansowej->wplywy = new \stdClass();
        $rachunek->przeplywySrodkowZDzialalnosciFinansowej->wplywy->razem = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciFinansowej->wplywy->zaciagniecieKredytow = $this->buildEmptyArrays();
		$rachunek->przeplywySrodkowZDzialalnosciFinansowej->wplywy->odsetkiOtrzymane = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciFinansowej->wydatki = new \stdClass();
        $rachunek->przeplywySrodkowZDzialalnosciFinansowej->wydatki->razem = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciFinansowej->wydatki->splatyKredytow = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciFinansowej->wydatki->odsetkiZaplacone = $this->buildEmptyArrays();
        $rachunek->przeplywySrodkowZDzialalnosciFinansowej->wydatki->inneWydatki = $this->buildEmptyArrays();

        $rachunek->przeplywyPieniezneRazem = $this->buildEmptyArrays();
        $rachunek->srodkiNaPoczatekOkresu = $this->buildEmptyArrays();
        $rachunek->srodkiNaKoniecOkresu = $this->buildEmptyArrays();

        $this->RachunekPrzeplywowPienieznych = $rachunek;
    }

    protected function evaluateRachunekPrzeplywowPienieznych()
    {
        $rachPrzeplywow = $this->RachunekPrzeplywowPienieznych;
        $rachZyskow = $this->RachunekZyskowIStrat;
        $aktywa = $this->Aktywa;
        $pasywa = $this->Pasywa;
        $inwestycje = $this->PodsumowanieFinansowaniaInwestycji->getPodsumowanieWszystkichInwestycji();

//        $this->evaluateRachunekPrzeplywowPrzeplywyZDzialalnosciOperacyjnej($rachPrzeplywow, $aktywa, $pasywa, $rachZyskow, $inwestycje);

        $this->evaluateRachunekPrzeplywowPrzeplywyZDzialalnosciInwestycyjnej($rachPrzeplywow, $aktywa, $inwestycje);

        $this->evaluateRachunekPrzeplywowPrzeplywyZDzialalnosciFinansowej($rachPrzeplywow, $inwestycje, $rachZyskow);

        $this->evaluateRachunekPrzeplywowPodsumowaniaOdDDoFZExcela($rachPrzeplywow);
    }

    protected function evaluateRachunekPrzeplywowZyskNetto($rachPrzeplywow, $aktywa, $pasywa, $rachZyskow, $inwestycje)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->zyskNetto[$i] = $rachZyskow->zyskNetto[$i];
        }
    }

    protected function evaluateRachunekPrzeplywowKorekty($rachPrzeplywow, $aktywa, $pasywa, $rachZyskow, $inwestycje)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->amortyzacja[$i] =
                isset($inwestycje->amortyzacjaNaRok[$i])? $inwestycje->amortyzacjaNaRok[$i]: 0;

            $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->zmianaStanuZapasow[$i] =
                $i == $this->rokPoczatkowy? - $aktywa->obrotowe->zapasy[$i]:
                    - ($aktywa->obrotowe->zapasy[$i] - $aktywa->obrotowe->zapasy[$i-1]);

            $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->zmianaStanuNaleznosci[$i] =
                $i == $this->rokPoczatkowy? - $aktywa->obrotowe->naleznosciKrotkoterminowe->razem[$i]:
                $aktywa->obrotowe->naleznosciKrotkoterminowe->razem[$i-1] - $aktywa->obrotowe->naleznosciKrotkoterminowe->razem[$i];

            $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->zmianaStanuZobowiazanKrotkoterminowych[$i] = 
                $i == $this->rokPoczatkowy? $pasywa->zobowiazania->krotkoterminowe->dostawyIUslugi[$i]:
                $pasywa->zobowiazania->krotkoterminowe->dostawyIUslugi[$i] - $pasywa->zobowiazania->krotkoterminowe->dostawyIUslugi[$i-1];

            $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->inneKorekty[$i] =
                $rachZyskow->kosztyFinansowe[$i] - $rachZyskow->przychodyFinansowe[$i];

            $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->razem[$i] =
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->amortyzacja[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->zmianaStanuZapasow[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->zmianaStanuNaleznosci[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->zmianaStanuZobowiazanKrotkoterminowych[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->inneKorekty[$i];
        }
    }

    protected function evaluateRachunekPrzeplywowPrzeplywyZDzialalnosciOperacyjnejRazem($rachPrzeplywow)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->razem[$i] =
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->zyskNetto[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->korekty->razem[$i];
        }
    }

    protected function evaluateRachunekPrzeplywowPrzeplywyZDzialalnosciInwestycyjnej($rachPrzeplywow, $aktywa, $inwestycje)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciInwestycyjnej->nakladyInwestycyjne[$i] =
                $i == $this->rokPoczatkowy? $aktywa->trwale->rzeczowe[$i]:
                    ($aktywa->trwale->rzeczowe[$i] - $aktywa->trwale->rzeczowe[$i-1] +
                        (isset($inwestycje->amortyzacjaNaRok[$i])? $inwestycje->amortyzacjaNaRok[$i]: 0));
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciInwestycyjnej->razem[$i] =
                -$rachPrzeplywow->przeplywySrodkowZDzialalnosciInwestycyjnej->nakladyInwestycyjne[$i];
        }
    }

    protected function evaluateRachunekPrzeplywowPrzeplywyZDzialalnosciFinansowej($rachPrzeplywow, $inwestycje, $rachZyskow)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $zmianaKapitaluObcegoNaRok =
                isset($inwestycje->zmianaKapitaluObcegoNaRok[$i])? $inwestycje->zmianaKapitaluObcegoNaRok[$i]: 0;

            $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wplywy->zaciagniecieKredytow[$i] =
                $zmianaKapitaluObcegoNaRok > 0? $zmianaKapitaluObcegoNaRok + $this->kredytObrotowy[$i]['value']: $this->kredytObrotowy[$i]['value'];
			$rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wplywy->odsetkiOtrzymane[$i] =
				$rachZyskow->przychodyFinansowe[$i];
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wplywy->inneWplywy[$i] = 
				$this->lokatySplacone[$i];
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wplywy->razem[$i] =
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wplywy->zaciagniecieKredytow[$i] +
				$rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wplywy->odsetkiOtrzymane[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wplywy->inneWplywy[$i];

            $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wydatki->splatyKredytow[$i] =
                $zmianaKapitaluObcegoNaRok < 0? -$zmianaKapitaluObcegoNaRok + $this->kredytObrotowySplata[$i]['fullValue']: +$this->kredytObrotowySplata[$i]['fullValue'];
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wydatki->odsetkiZaplacone[$i] = 
				$rachZyskow->kosztyFinansowe[$i];
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wydatki->inneWydatki[$i] = 
				$this->lokatyWziete[$i];
            $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wydatki->razem[$i] =
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wydatki->splatyKredytow[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wydatki->odsetkiZaplacone[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wydatki->inneWydatki[$i];

            $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->razem[$i] =
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wplywy->razem[$i] -
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->wydatki->razem[$i];
        }
    }

    protected function evaluateRachunekPrzeplywowPodsumowaniaOdDDoFZExcela($rachPrzeplywow)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $rachPrzeplywow->przeplywyPieniezneRazem[$i] =
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciOperacyjnej->razem[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciInwestycyjnej->razem[$i] +
                $rachPrzeplywow->przeplywySrodkowZDzialalnosciFinansowej->razem[$i];

            $rachPrzeplywow->srodkiNaPoczatekOkresu[$i] =
                $i == $this->rokPoczatkowy? $this->kapitalPodstawowy: $rachPrzeplywow->srodkiNaKoniecOkresu[$i-1];
            $rachPrzeplywow->srodkiNaKoniecOkresu[$i] =
                $rachPrzeplywow->przeplywyPieniezneRazem[$i] +
                $rachPrzeplywow->srodkiNaPoczatekOkresu[$i];
        }
    }

    protected function buildWskaznikiFinansowe()
    {
        $wskazniki = new \stdClass();

        $wskazniki->plynnosci = new \stdClass();
        $wskazniki->plynnosci->biezaca = $this->buildEmptyArrays();
        $wskazniki->plynnosci->szybka = $this->buildEmptyArrays();

        $wskazniki->sprawnosciDzialania = new \stdClass();
        $wskazniki->sprawnosciDzialania->rotacjaZapasow = $this->buildEmptyArrays();
        $wskazniki->sprawnosciDzialania->rotacjaNaleznosci = $this->buildEmptyArrays();
        $wskazniki->sprawnosciDzialania->rotacjaZobowiazan = $this->buildEmptyArrays();

        $wskazniki->strukturyKapitalu = new \stdClass();
        $wskazniki->strukturyKapitalu->zadluzenieKapitaluWlasnego = $this->buildEmptyArrays();
        $wskazniki->strukturyKapitalu->poziomZadluzeniaOgolnego = $this->buildEmptyArrays();

        $wskazniki->rentownosci = new \stdClass();
        $wskazniki->rentownosci->sprzedazyNetto = $this->buildEmptyArrays();
        $wskazniki->rentownosci->kapitaluWlasnegoROE = $this->buildEmptyArrays();
        $wskazniki->rentownosci->majatkuROA = $this->buildEmptyArrays();

        $this->WskaznikiFinansowe = $wskazniki;
    }

    protected function evaluateWskaznikiFinansowe($wskazniki, $aktywa, $pasywa, $rachZyskow)
    {
        for($i = $this->rokPoczatkowy; $i < $this->doKtoregoRoku; $i++)
        {
            $wskazniki->plynnosci->biezaca[$i] =
                $i == $this->rokPoczatkowy? 0:
                $this->division($aktywa->obrotowe->razem[$i], $pasywa->zobowiazania->krotkoterminowe->razem[$i]);
            $wskazniki->plynnosci->szybka[$i] =
                $i == $this->rokPoczatkowy? 0:
                $this->division(($aktywa->obrotowe->razem[$i] - $aktywa->obrotowe->zapasy[$i]), $pasywa->zobowiazania->krotkoterminowe->razem[$i]);

            $wskazniki->sprawnosciDzialania->rotacjaZapasow[$i] = $this->rotacjaZapasowWDniach;
            $wskazniki->sprawnosciDzialania->rotacjaNaleznosci[$i] = $this->rotacjaNaleznosciWDniach;
            $wskazniki->sprawnosciDzialania->rotacjaZobowiazan[$i] = $this->rotacjaZobowiazanWDniach;

            $wskazniki->strukturyKapitalu->zadluzenieKapitaluWlasnego[$i] =
                $i == $this->rokPoczatkowy? 0:
                $this->division($pasywa->zobowiazania->razem[$i], $pasywa->kapitalWlasny->razem[$i]);
            $wskazniki->strukturyKapitalu->poziomZadluzeniaOgolnego[$i] = $i == $this->rokPoczatkowy? 0:
                $this->division($pasywa->zobowiazania->razem[$i], $pasywa->razem[$i]);

            $wskazniki->rentownosci->sprzedazyNetto =
                $i == $this->rokPoczatkowy? 0:
                $this->division($rachZyskow->zyskNetto[$i], $rachZyskow->przychodyNettoZeSprzedazy->razem[$i]);
            $wskazniki->rentownosci->kapitaluWlasnegoROE =
                $i == $this->rokPoczatkowy? 0:
                $this->division($rachZyskow->zyskNetto[$i], $pasywa->kapitalWlasny->razem[$i]);
            $wskazniki->rentownosci->majatkuROA =
                $i == $this->rokPoczatkowy? 0:
                $this->division($rachZyskow->zyskNetto[$i], $pasywa->razem[$i]);
        }
    }

    protected function evaluateAll()
    {
        $rachZyskow = $this->RachunekZyskowIStrat;
        $aktywa = $this->Aktywa;
        $pasywa = $this->Pasywa;
        $rachPrzeplywow = $this->RachunekPrzeplywowPienieznych;
        $inwestycje = $this->PodsumowanieFinansowaniaInwestycji->getPodsumowanieWszystkichInwestycji();
        $wskazniki = $this->WskaznikiFinansowe;

//        echo "<pre>";
//        print_r($inwestycje);

        // WAŻNA JEST KOLEJNOŚĆ WYKONYWANIA METOD!!!
        $this->evaluateRachunekZyskowKosztyDzialalnosciOperacyjnej($rachZyskow, $inwestycje);

        $this->evaluateAktywaObrotoweZapasy($aktywa, $rachZyskow, $rachPrzeplywow);

        $this->evaluateRachunekZyskowPrzychodyNettoZeSprzedazy($aktywa, $rachZyskow);

        $this->evaluateRachunekZyskowResztaOdCDoLZExcela($rachZyskow, $inwestycje);

        $this->evaluateAktywaTrwale($aktywa, $inwestycje);

        $this->evaluatePasywaKapitalWlasny($pasywa, $rachZyskow);

        $this->evaluatePasywaZobowiazania($pasywa, $rachZyskow, $inwestycje, $aktywa);

        $this->evaluatePasywaRazem($pasywa);

        $this->evaluateRachunekPrzeplywowPrzeplywyZDzialalnosciInwestycyjnej($rachPrzeplywow, $aktywa, $inwestycje);

        $this->evaluateAktywaObrotoweNaleznosci($aktywa, $rachZyskow, $rachPrzeplywow);

        $this->evaluateRachunekPrzeplywowZyskNetto($rachPrzeplywow, $aktywa, $pasywa, $rachZyskow, $inwestycje);

        $this->evaluateRachunekPrzeplywowKorekty($rachPrzeplywow, $aktywa, $pasywa, $rachZyskow, $inwestycje);

        $this->evaluateRachunekPrzeplywowPrzeplywyZDzialalnosciOperacyjnejRazem($rachPrzeplywow, $aktywa, $pasywa, $rachZyskow, $inwestycje);

        $this->evaluateRachunekPrzeplywowPrzeplywyZDzialalnosciFinansowej($rachPrzeplywow, $inwestycje, $rachZyskow);

        $this->evaluateRachunekPrzeplywowPodsumowaniaOdDDoFZExcela($rachPrzeplywow);

        $this->evaluateAktywaObrotoweInwestycjeKrotkoterminowe($aktywa, $rachPrzeplywow);

        $this->evaluateAktywaObrotoweRazem($aktywa);

        $this->evaluateAktywaRazem($aktywa, $pasywa);

        $this->evaluateWskaznikiFinansowe($wskazniki, $aktywa, $pasywa, $rachZyskow);
    }

    public function getResultsOf($name)
    {
        return $this->$name;
    }

    public function getAll()
    {
        return array('Aktywa' => $this->Aktywa, 'Pasywa' => $this->Pasywa, 'RachunekZyskowIStrat' => $this->RachunekZyskowIStrat, 'RachunekPrzeplywowPienieznych' => $this->RachunekPrzeplywowPienieznych);
    }

    public function getProductionData()
    {
        $iteracja = $this->FinansowanieProdukcji->getResultsOf('iteracja');
        $result = array();
        $result['wyprodukowano'] = $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($iteracja)->produkcjaLacznie;
        $result['sprzedano'] = $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($iteracja)->sprzedazLacznie;
        $result['zapasy'] = $this->FinansowanieProdukcji->getWariantProporcjonalnyByYear($iteracja)->zapasy;
        $result['podatekDochodowy'] = $this->RachunekZyskowIStrat->podatekDochodowy[$iteracja];
        $result['naleznosci'] = $this->Aktywa->obrotowe->naleznosciKrotkoterminowe->razem[$iteracja];
		$result['naleznosciStare'] = 
			$iteracja == $this->rokPoczatkowy? 0 : $this->Aktywa->obrotowe->naleznosciKrotkoterminowe->razem[$iteracja-1];
        $result['zobowiazania'] = $this->Pasywa->zobowiazania->krotkoterminowe->razem[$iteracja];
		$result['zobowiazaniaStare'] = 
			$iteracja == $this->rokPoczatkowy? 0 : $this->Pasywa->zobowiazania->krotkoterminowe->razem[$iteracja-1];
        return $result;
    }
}

?>